/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE mesh_deform.c
HEADER mesh_deform.h

TO_HEADER:
*/


#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "assembly.h"
#include "gen_aux.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

#include <unistd.h>
#include <string.h>

/* #define K_LINE_DENSE 27 */
#define K_LINE_DENSE 35

/*FUNCTION*/
int mesh_deform_t1_t2(struct mesh *msh1, FIDX lvlm,
		      double **vx_new, FIDX *vx_max,
		      FIDX nI, struct vector *dIdx
/* computes a deformation of refined T2 versions of T1 mesh msh1,
   stores the resulting node positions in vx_new (if vx_new!=NULL)
   OR
   corrects dIdx to take into account that the position of the
   interior nodes are only a function of the nodes on the boundary
   (if dIdx!=NULL)

   Input:  msh1    - the coarse mesh (T1), hierarchy has to be reset,
                     i.e. msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1
		     for all j
           lvlm    - the maximal refinement level
	   nI      - size of dIdx (number of criteria)
           
   Output: vx_new  - reference to a pointer (vector) containing the
                     node positions for the finest mesh (T2), for
                     connectivity as resulting from mesh_t1_to_t2 and
                     then lvlm uniform refinements of the T2 mesh,
                     this vector can be straight plugged into each
                     mesh derived this way by     msh2.vertex=vx_new
                     so long as msh2.vx_nr<=vx_max, memory is
                     allocated here and has to be freed by the calling
                     routine, ignored if (vx_new==NULL)
	   vx_max  - number of nodes in vx_new (by reference), ignored
	             if (vx_new==NULL)

   In/Out: dIdx    - array of nI vectors of lentgh dim*vx_nr (finest
                     mesh), such that dIdx[i].V[d*vx_nr+j] is the
                     derivative of the i-th criterion wrt the
                     d-coordinate of the j-th node, if (dIdx!=NULL),
                     the values are corrected to reflect that the
                     position of the interior nodes is a function of
                     the position of the boundary nodes

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
	   10      - element orientation or area changed too much,
	             (detA_new/detA_old < 0.1)
*/
		      ){

  struct sparse *Ks_linel;
  struct projector1 P_linel;

  struct vector rhs_linel, nodem, p_res;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;

  double *elem_detA_old, *elem_detA_new;
  double maxal_old, maxal_new;
  int test_OK;

  FIDX dim, vx_nr, i, d;
  int  err;

  FIDX lvl;

  double mu, lambda;
  int stiffening;

  /* Lame canstants for steel */
  mu         = 7.7e4;  /* 7.7e4; */
  lambda     = 1.15e5; /* 1.15e5; */
  stiffening = 0;


  dim=msh1->dim;

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, mesh_deform_t1_t2);

  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks_linel,  lvlm+1, struct sparse, mesh_deform_t1_t2);
  
  /* assemble the coarse meshes + refine */
  for (lvl=0; lvl<lvlm; lvl++)
    {
      vx_nr=msh2.vx_nr; 

      err=sparse_alloc( &Ks_linel[lvl], dim*vx_nr, dim*K_LINE_DENSE);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, mesh_deform_t1_t2);

      err=assem_lame_tx( &Ks_linel[lvl], NULL, &msh2, NULL, NULL,
			 lambda, mu, stiffening, 1, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, mesh_deform_t1_t2);

      err=mesh_refine_uniform_t2( &msh2 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2,
			       mesh_deform_t1_t2);
    }

  vx_nr=msh2.vx_nr; 


  err=sparse_alloc( &Ks_linel[lvlm], dim*vx_nr, dim*K_LINE_DENSE);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, mesh_deform_t1_t2);

  err=projector1_alloc( &P_linel, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, mesh_deform_t1_t2);

  /* assemble the system on the fine mesh*/
  err=assem_lame_tx( &Ks_linel[lvlm], &P_linel, &msh2, NULL, NULL,
		     lambda, mu, stiffening, 1, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, mesh_deform_t1_t2);

  /* define the multigrid data */
  err=multilvl_init_tx( &msh2, dim, &ml, 2);
  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, mesh_deform_t1_t2);

  err=mg_init_tx( Ks_linel, &msh2, &ml, &mg, &P_linel);
  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, mesh_deform_t1_t2);


  if (vx_new!=NULL)
    {
      /* write the initial mesh */
      struct vector step;
      FILE *mark;

      err=vector_alloc(&step, vx_nr);
      FUNCTION_FAILURE_HANDLE(err, vector_alloc, mesh_deform_t1_t2);

      for (i=0; i<step.len; i++)
	step.V[i]=0.0;
      err=mesh_write_solution_exp_t2( &msh2, &step, 1, 19,
				      "visual/2d_new_mesh");
      FUNCTION_FAILURE_HANDLE(err, mesh_write_solution_exp_t2,
			      mesh_deform_t1_t2);

      err=mesh_write_mesh_gnuplot_tx( &msh2, 17, "visual/gnup_mesh",2);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			      mesh_deform_t1_t2);

      /* touch the marker file */
      mark=fopen("visual/2d_new_mesh_mark", "w");
      fprintf(mark, "next please!\n");
      fclose(mark);
      vector_free(&step);
    }


  /* generate the coarse grid matrix */
  {
    struct coarse_mat *cmat;
    FIDX *coarse_bc_nodes;
    FIDX n_c_bc=0;

    TRY_MALLOC(cmat, 1,  struct coarse_mat, mesh_deform_t1_t2);
    TRY_MALLOC(coarse_bc_nodes, P_linel.len, FIDX, mesh_deform_t1_t2);
     
    for (i=0; i<P_linel.len; i++)
      {
	FIDX dof, child;
	dof=P_linel.V[i];
	MLVLFINDENTRY(child, dof, 0, ml);
	if (child>=0)
	  {
	    coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
	    n_c_bc++;
	  }
      }
    /* printf("n_c_bc=%d\n", (int) n_c_bc); /* */
    /* sparse_mat_write_file( &Ks_linel[0], "visual/Klinel.txt"); /* */
fprintf( stderr,"2\n");

    err=coarse_mat_set( &Ks_linel[0], n_c_bc, coarse_bc_nodes,
			1, cmat );
    FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, mesh_deform_t1_t2 );
    mg.cmat=cmat;
fprintf( stderr,"3\n");
    free(coarse_bc_nodes);
  } /* */

  /* define solver parameter for the V-cycle */
  mg.vcycles=1;
  mg.smooths=1;
  if (stiffening==0)
    mg.CGC_scale=0.25;
  else
    mg.CGC_scale=1.0;

  err=vector_alloc( &rhs_linel, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, mesh_deform_t1_t2);

  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, mesh_deform_t1_t2);

  err=vector_alloc( &p_res, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, mesh_deform_t1_t2);
;
  if (vx_new!=NULL)
    {
      double *vx_tmp, *vx_save;
      int iter;
      double resi;
      double tollin;

      /* set vectors to zero */
      for (i=0; i<dim*vx_nr; i++)
	{
	  nodem.V[i]    =0.0;
	  rhs_linel.V[i]=0.0;
	}

      /* now define the BC-data for the linear-elasticity problem */
      TRY_MALLOC(vx_tmp, msh2.vx_w*msh2.vx_nr, double, mesh_deform_t1_t2);
      vx_save=msh2.vertex;
  
      for (i=0; i<msh2.vx_w*msh2.vx_nr; i++)
	vx_tmp[i]=vx_save[i];
    
      msh2.vertex=vx_tmp;

      /*                        msh, insert, nI, dIdx, dIdF, type */
      err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, mesh_deform_t1_t2);
      msh2.vertex=vx_save;

      for (i=0; i<P_linel.len; i++)
	{
	  FIDX dof, node, ndim;
	  double xy_old, xy_new;

	  dof  = P_linel.V[i];
	  node = dof%vx_nr;
	  ndim = dof/vx_nr;

	  xy_old = msh2.vertex[node*msh2.vx_w+MCT2VXSTRT+ndim];
	  xy_new =      vx_tmp[node*msh2.vx_w+MCT2VXSTRT+ndim];

	  nodem.V[dof]=xy_new-xy_old;

	  /* this resulted in problems with accuracy on highly
	     anisotropic meshes */
	  if (fabs(nodem.V[dof])<1e-14)
	    nodem.V[dof]=0.0;
	}

      free(vx_tmp);

      /* end setting boundary data */

      /* compute the norm of the residual */
      /* rhs_linel = res= K*nodem-0 */
      err=sparse_mul_mat_vec( &Ks_linel[lvlm], &nodem, &rhs_linel);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec,
			       mesh_deform_t1_t2);  
      /* preconditioned residual */
      err=gen_proj_MG_tx( &Ks_linel[lvlm], &rhs_linel, &mg, &p_res);
      FUNCTION_FAILURE_HANDLE( err, PCG, mesh_deform_t1_t2);

      resi=0.0;
      for (i=0; i<dim*vx_nr; i++)
	resi+= p_res.V[i]*p_res.V[i];
      resi=sqrt(resi);  
      /* set rhs_linel to zero, again*/
      for (i=0; i<dim*vx_nr; i++)
	rhs_linel.V[i]=0.0;
      	  
      /* as we use relative residual reduction, we have to take
	 care we don't run into trouble with accuracy */
#define MAX_ACCURACY_ELAST_ABS (1.0e-14)
#define MAX_ACCURACY_ELAST_REL (1.0e-6)
      tollin=MAX_ACCURACY_ELAST_REL;
      if ((tollin*resi<MAX_ACCURACY_ELAST_ABS)&&(resi>0.0))
	tollin=(MAX_ACCURACY_ELAST_ABS)/resi;

      printf("|rhs|=%8.1e  tollin=%8.1e \n",resi, tollin);


      /* solve the equation system */

      /* rhs is just zero, use nodem directly */
      err=PCG( 500, 2, tollin, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks_linel[lvlm], &rhs_linel, &mg); /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, mesh_deform_t1_t2);
    
      printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
    
  
      /* before we apply the node movement, store the old element
	 orientation, so we can later test for changes */
      TRY_MALLOC( elem_detA_old, msh2.el_nr, double, mesh_deform_t1_t2);
      err=assem_elem_detA_tx( &msh2, elem_detA_old, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_elem_detA_tx, mesh_deform_t1_t2);

      err=max_angle_tx( &msh2, &maxal_old, 2);
      FUNCTION_FAILURE_HANDLE( err, max_angle_tx, mesh_deform_t1_t2);
      
      /* the node movement computation is done, allpy it to the mesh */
      for (i=0; i<vx_nr; i++)
	for (d=0; d<dim; d++)
	  {
	    msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += nodem.V[d*vx_nr+i];
	  }

      /* now get the new element orientation */
      TRY_MALLOC( elem_detA_new, msh2.el_nr, double, mesh_deform_t1_t2);
      err=assem_elem_detA_tx( &msh2, elem_detA_new, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_elem_detA_tx, mesh_deform_t1_t2);

      err=max_angle_tx( &msh2, &maxal_new, 2);
      FUNCTION_FAILURE_HANDLE( err, max_angle_tx, mesh_deform_t1_t2);

      printf("mesh_deform_t1_t2: "
	     "maxal_new=%f degree, maxal_old=%f degree\n",
	     maxal_new*180.0/M_PI,maxal_old*180.0/M_PI);
      /* test for changes */
      if (maxal_new > 145.0/180.0*M_PI)
	{
	  fprintf(stderr,"mesh_deform_t1_t2: "
		  "maxal_new=%f  (%f degree) too large, maxal_old=%f\n",
		  maxal_new, maxal_new*180.0/M_PI,maxal_old);
	  test_OK=0;
	}
      else
	test_OK=1;
      for (i=0; (i<msh2.el_nr)&&(test_OK); i++)
	{
	  if ( elem_detA_new[i]/elem_detA_old[i] < 0.1 )
	    {
	      fprintf(stderr,"mesh_deform_t1_t2: "
		      "detA_new[%d]/detA_old[%d]=%e\n", (int) i, (int) i,
		      elem_detA_new[i]/elem_detA_old[i]);
	      test_OK=0;
	    }
	}

      {
	/* write the resulting mesh */
	struct vector step;
	FILE *mark;

	err=vector_alloc(&step, vx_nr);
	FUNCTION_FAILURE_HANDLE(err, vector_alloc, mesh_deform_t1_t2);

	for (i=0; i<step.len; i++)
	  step.V[i]=0.0;
	err=mesh_write_solution_exp_t2( &msh2, &step, 1, 19,
					"visual/2d_new_mesh");
	FUNCTION_FAILURE_HANDLE(err, mesh_write_solution_exp_t2,
				mesh_deform_t1_t2);

	err=mesh_write_mesh_gnuplot_tx( &msh2, 17, "visual/gnup_mesh",2);
	FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
				mesh_deform_t1_t2);

	err=mesh_write_bound_gnuplot_tx( &msh2, 18, "visual/gnup_bound",2);
	FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
				mesh_deform_t1_t2);

	/* system("gnuplot plot_bound.gp"); exit(1); /* */

	/* touch the marker file */
	mark=fopen("visual/2d_new_mesh_mark", "w");
	fprintf(mark, "next please!\n");
	fclose(mark);
	vector_free(&step);
      }

      /* define the output */
      *vx_max = msh2.vx_nr;
      *vx_new  = msh2.vertex;
      msh2.vertex = NULL; /* protect vx_new from being freed */

      /* free locla data */
      free(elem_detA_new);
      free(elem_detA_old);

    } /* end move nodes if (vx_new!=NULL) */

  if (dIdx!=NULL) /* derivative correction */
    {
      FIDX tc;
      int iter;
      double resi, tollin;

      for(tc=0; tc<nI; tc++)
	{

	  /* set nodem to zero, rhs to dIdx */
	  for (i=0; i<dim*vx_nr; i++)
	    {
	      nodem.V[i]    =0.0;
	      rhs_linel.V[i]=dIdx[tc].V[i];
	    }

	  /* set rhs to zero on boundary */
	  for (i=0; i<P_linel.len; i++)
	    {
	      FIDX dof = P_linel.V[i];
	      rhs_linel.V[dof]=0.0;
	    }/* */
	  err=gen_proj_MG_tx( &Ks_linel[lvlm], &rhs_linel, &mg, &p_res);
	  FUNCTION_FAILURE_HANDLE( err, PCG, mesh_deform_t1_t2);
	  /* end setting rhs data */

	  /* compute the norm of the rhs */
	  resi=0.0;
	  for (i=0; i<dim*vx_nr; i++)
	    resi+= p_res.V[i]*p_res.V[i];
	  resi=sqrt(resi);  
	  
	  /* as we use relative residual reduction, we have to take
	     care we don't run into trouble with accuracy */
	  tollin=MAX_ACCURACY_ELAST_REL;
	  if ((tollin*resi<MAX_ACCURACY_ELAST_ABS)&&(resi>0.0))
	    tollin=(MAX_ACCURACY_ELAST_ABS)/resi;
#undef MAX_ACCURACY_ELAST_ABS
#undef MAX_ACCURACY_ELAST_REL

	  printf("|rhs|=%8.1e  tollin=%8.1e \n",resi, tollin);

	  /* solve the equation system */
	  err=PCG( 500, 2, tollin, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_MG_tx,
		   &Ks_linel[lvlm], &rhs_linel, &mg); /* */
	  printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
	  FUNCTION_FAILURE_HANDLE( err, PCG, mesh_deform_t1_t2);
    


	  /* apply the result */
	  /*         y   := A_ii^-T dIdx_i  is in nodem */
	  /* correction  :=-A_bi*y          build A_bi*y rhs */
	  err=sparse_mul_mat_vec( &Ks_linel[lvlm], &nodem, &rhs_linel);
	  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec,
				   mesh_deform_t1_t2);  
	  /* on the boundary this is now A_bi*y */
	  
	  /* set nodem_b = dIdx -A_bi*y, nodem_i = 0 */
	  for (i=0; i<dim*vx_nr; i++)
	    nodem.V[i]=0.0;
	  for (i=0; i<P_linel.len; i++)
	    {
	      FIDX dof = P_linel.V[i];
	      nodem.V[dof]=dIdx[tc].V[dof]-rhs_linel.V[dof];
	    }
	  
	  /* copy the result to dIdx */
	  for (i=0; i<dim*vx_nr; i++)
	    dIdx[tc].V[i]=nodem.V[i];

	} /* end loop criteria tc=0..nI */

      test_OK=1;
    } /* end derivative correction */

  mg_free(&mg);
  multilvl_free(&ml);

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks_linel[lvl]);
  free(Ks_linel);

  projector1_free(&P_linel);
  
  vector_free(&rhs_linel);
  vector_free(&nodem);
  vector_free(&p_res);

  mesh_free(&msh2);

  if (test_OK)
    return SUCCESS;
  else
    return 10;
}
