/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE elements.c
HEADER elements.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"

*/



/* function prototypes */
#include "elements.h"

/*FUNCTION*/
int eval_basis(FIDX dim, enum elemtype type, FIDX subtype, 
	       FIDX num_points, double points[], 
	       FIDX *num_basis, double **phi, double **gradphi,
	       double **hessphi
/* evaluate the basisfunctions phi and their gradients gradphi for the
   element type type at the given points points (which have to be in
   the appropriate masterelement) 

   Input:  dim    - dimension of space
           type   - type of the element (e.g. tria, see element types)
	   subtype- sub-type of the element, e.g. 1 for P_1 triangles,
 	            while 2 for P_2 triangles
	   num_points
	          - number of points for which the evaluation is wanted
	   points - array of dimension [num_points*dim] containing the
	            points where the basis function has to evaluated,
		    usage: points[i*dim+j] = j-th component of i-th point

   Output: num_basis
	          - number of basis functions for this element type,
		    via reference
	   phi    - pointer to array of dimension [num_points*num_basis]
                    containing the j-th basis function evaluated at
                    the i-th point as phi[i*num_basis+j],
		    via reference, array is created
	   gradphi- pointer to array of dimension [num_points*num_basis*dim]
                    containing the gradient of phi,
		    gradphi[i*num_basis*dim +j*dim +k],
		    the k-th component of the gradient of the j-th
		    basis function evaluated at the i-th point,
		    via reference, array is created
	   hessphi- pointer to array of dimension
	            [num_points*num_basis*dim*dim] containing the
	            Hessian of phi, 
		    hessphi[i*num_basis*dim*dim +j*dim*dim +k*dim +l],
		    is the derivative of the j-th basis function with
		    respect to the k-th and l-th spacial dimension,
		    evaluated at the i-th point, provided via
		    reference 

   Return: SUCCESS- succes
           FAIL   - failure, see error message
*/
	       ){
  FIDX i,j;

  /* for each basis type blocks like:
       elseif ((type==)&&(subtype==))
         {
            set to something;
            return SUCCESS;
         }
       elseif ... */
  if (((type==inter)&&(subtype==1))&&(dim==1))
    {
      /* linear on the interval [0,1] */
      *num_basis=2;
      TRY_MALLOC( *phi, num_points*(*num_basis), double,
		  eval_basis ); 
      TRY_MALLOC( *gradphi, num_points*(*num_basis)*dim, double,
		  eval_basis );
      TRY_MALLOC( *hessphi, num_points*(*num_basis)*dim*dim, double,
		  eval_basis );
      /* evaluate for each point */
      for(i=0; i<num_points; i++)
	{
	  /* phi_0=1-x, phi_1=x, */
	  (*phi)[i*(*num_basis)+0] = 1.0 - points[i*dim +0];
	  (*phi)[i*(*num_basis)+1] = points[i*dim +0];
	  
	  /* grad(phi_0)= -1; (indepenent of the point) */
	  (*gradphi)[i*(*num_basis)*dim           ] = -1.0;
	  /* grad(phi_1)=  1; (indepenent of the point) */
	  (*gradphi)[i*(*num_basis)*dim+   dim    ] =  1.0;

	  /* linear elements have vanishing Hessian */
	  for (j=0; j<(*num_basis)*dim*dim; j++)
	    (*hessphi)[i*(*num_basis)*dim*dim+j]=0.0;
	}
      return SUCCESS;
    }
  else if (((type==inter)&&(subtype==2))&&(dim==1))
    {
      /* quadratic on the interval [0,1]
	 corresponding to the node numbering
	 0--2--1 (node numbers)
	 0 0.5 1 (x)
      */
      *num_basis=3;
      TRY_MALLOC( *phi, num_points*(*num_basis), double,
		  eval_basis ); 
      TRY_MALLOC( *gradphi, num_points*(*num_basis)*dim, double,
		  eval_basis );
      TRY_MALLOC( *hessphi, num_points*(*num_basis)*dim*dim, double,
		  eval_basis );

      /* evaluate for each point */
      for(i=0; i<num_points; i++)
	{
#define X (points[i*dim +0])
	  /* phi_0=2x^2-3x+1, phi_1=2x^2-x, phi_2=-4x^2+4x */
	  (*phi)[i*(*num_basis)+0] = (2*X-3)*X+1;
	  (*phi)[i*(*num_basis)+1] = (2*X-1)*X;
	  (*phi)[i*(*num_basis)+2] = (-4*X+4)*X;
	  
	  /* grad(phi_0)= 4x-3 */
	  (*gradphi)[i*(*num_basis)*dim           ] = 4*X-3;
	  /* grad(phi_1)= 4x-1 */
	  (*gradphi)[i*(*num_basis)*dim+   dim    ] = 4*X-1;
	  /* grad(phi_2)= -8x+4 */
	  (*gradphi)[i*(*num_basis)*dim+ 2*dim    ] = -8*X+4;

	  /* d^2 phi_1 / dx^2 = 4  */
	  (*hessphi)[i*(*num_basis)*dim*dim           ] = 4;
	  /* d^2 phi_2 / dx^2 = 4  */
	  (*hessphi)[i*(*num_basis)*dim*dim+   dim*dim] = 4;
	  /* d^2 phi_3 / dx^2 = -8 */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 2*dim*dim] = -8;
#undef X
	}
      return SUCCESS;
    }
  else if (((type==tria)&&(subtype==1))&&(dim==2))
    {
      *num_basis=3;
      TRY_MALLOC( *phi, num_points*(*num_basis), double,
		  eval_basis ); 
      TRY_MALLOC( *gradphi, num_points*(*num_basis)*dim, double,
		  eval_basis );
      TRY_MALLOC( *hessphi, num_points*(*num_basis)*dim*dim, double,
		  eval_basis );
      /* evaluate for each point */
      for(i=0; i<num_points; i++)
	{
	  /* phi_0=1-x-y, phi_1=x, phi_2=y */
	  (*phi)[i*(*num_basis)+0] = 1.0 - points[i*dim +0] - points[i*dim +1];
	  (*phi)[i*(*num_basis)+1] = points[i*dim +0];
	  (*phi)[i*(*num_basis)+2] = points[i*dim +1];
	  
	  /* grad(phi_0)= [-1, -1]; (indepenent of the point) */
	  (*gradphi)[i*(*num_basis)*dim           ] = -1.0;
	  (*gradphi)[i*(*num_basis)*dim        + 1] = -1.0;
	  /* grad(phi_1)= [ 1,  0]; (indepenent of the point) */
	  (*gradphi)[i*(*num_basis)*dim+   dim + 0] =  1.0;
	  (*gradphi)[i*(*num_basis)*dim+   dim + 1] =  0.0;
	  /* grad(phi_2)= [ 0,  1]; (indepenent of the point) */
	  (*gradphi)[i*(*num_basis)*dim+ 2*dim + 0] =  0.0;
	  (*gradphi)[i*(*num_basis)*dim+ 2*dim + 1] =  1.0;

	  /* linear elements have vanishing Hessian */
	  for (j=0; j<(*num_basis)*dim*dim; j++)
	    (*hessphi)[i*(*num_basis)*dim*dim+j]=0.0;
	}
      return SUCCESS;
    }
  else if (((type==tria)&&(subtype==2))&&(dim==2))
    {
      /* numbering of the points in the masterelement:

         2
	 |\
	 5 4
	 |  \
	 0-3-1

	 the basis functions correspond to this numbering
      */
      *num_basis=6;
      TRY_MALLOC( *phi, num_points*(*num_basis), double,
		  eval_basis ); 
      TRY_MALLOC( *gradphi, num_points*(*num_basis)*dim, double,
		  eval_basis );
      TRY_MALLOC( *hessphi, num_points*(*num_basis)*dim*dim, double,
		  eval_basis );
      /* evaluate for each point */
      for(i=0; i<num_points; i++)
	{
	  /* let us define wsomething to make it more readable: */
#define X (points[i*dim +0])
#define Y (points[i*dim +1])
	  /* the basis functions */
	  (*phi)[i*(*num_basis)+0] = (2*X+4*Y-3)*X+(2*Y-3)*Y+1;
	  (*phi)[i*(*num_basis)+1] = X*(2*X-1);
	  (*phi)[i*(*num_basis)+2] = Y*(2*Y-1);
	  (*phi)[i*(*num_basis)+3] = 4*X*(1-X-Y);
	  (*phi)[i*(*num_basis)+4] = 4*X*Y;
	  (*phi)[i*(*num_basis)+5] = 4*Y*(1-X-Y);
	  
	  /* grad(phi_0) */
	  (*gradphi)[i*(*num_basis)*dim           ] = 4*X+4*Y-3;
	  (*gradphi)[i*(*num_basis)*dim        + 1] = 4*Y+4*X-3;
	  /* grad(phi_1) */
	  (*gradphi)[i*(*num_basis)*dim+   dim + 0] = 4*X-1;
	  (*gradphi)[i*(*num_basis)*dim+   dim + 1] = 0.0;
	  /* grad(phi_2) */
	  (*gradphi)[i*(*num_basis)*dim+ 2*dim + 0] = 0.0;
	  (*gradphi)[i*(*num_basis)*dim+ 2*dim + 1] = 4*Y-1;
	  /* grad(phi_3) */
	  (*gradphi)[i*(*num_basis)*dim+ 3*dim + 0] = -8*X-4*Y+4;
	  (*gradphi)[i*(*num_basis)*dim+ 3*dim + 1] = -4*X;
	  /* grad(phi_4) */
	  (*gradphi)[i*(*num_basis)*dim+ 4*dim + 0] = 4*Y;
	  (*gradphi)[i*(*num_basis)*dim+ 4*dim + 1] = 4*X;
	  /* grad(phi_5) */
	  (*gradphi)[i*(*num_basis)*dim+ 5*dim + 0] = -4*Y;
	  (*gradphi)[i*(*num_basis)*dim+ 5*dim + 1] = -8*Y-4*X+4;

	  /* Hessian(phi_0) */
	  /* d^2 phi_0 / (dx dx) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 0*dim*dim      + 0] = 4;
	  /* d^2 phi_0 / (dx dy) */
	  (*hessphi)[i*(*num_basis)*dim*dim+                + 1] = 4;
	  /* d^2 phi_0 / (dy dx) */
	  (*hessphi)[i*(*num_basis)*dim*dim+            dim + 0] = 4;
	  /* d^2 phi_0 / (dy dy) */
	  (*hessphi)[i*(*num_basis)*dim*dim+            dim + 1] = 4;

	  /* Hessian(phi_1) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 1*dim*dim      + 0] = 4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 1*dim*dim      + 1] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 1*dim*dim+ dim + 0] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 1*dim*dim+ dim + 1] = 0;
	  
	  /* Hessian(phi_2) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 2*dim*dim      + 0] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 2*dim*dim      + 1] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 2*dim*dim+ dim + 0] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 2*dim*dim+ dim + 1] = 4;
	  
	  /* Hessian(phi_3) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 3*dim*dim      + 0] = -8;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 3*dim*dim      + 1] = -4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 3*dim*dim+ dim + 0] = -4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 3*dim*dim+ dim + 1] = 0;
	  
	  /* Hessian(phi_4) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 4*dim*dim      + 0] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 4*dim*dim      + 1] = 4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 4*dim*dim+ dim + 0] = 4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 4*dim*dim+ dim + 1] = 0;
	  
	  /* Hessian(phi_5) */
	  (*hessphi)[i*(*num_basis)*dim*dim+ 5*dim*dim      + 0] = 0;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 5*dim*dim      + 1] = -4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 5*dim*dim+ dim + 0] = -4;
	  (*hessphi)[i*(*num_basis)*dim*dim+ 5*dim*dim+ dim + 1] = -8;
	  
#undef X
#undef Y
	}
      return SUCCESS;
    }
  else
    {
      fprintf(stderr,
	      "sorry, eval_basis not implemented for\n(%d,%d,*)\n",
	      (int) dim, type);
      return FAIL;
    }
}
