/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
assembly.h
*/
#ifndef __ASSEMBLY_H__
#define __ASSEMBLY_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"


int assem_poison_t1(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P, 
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poison equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poison equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (only p1 triangles for now)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   u0      - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_poison_t2(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P,
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poison equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poison equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   u0      - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_poison_t1_t2(struct sparse *K, FIDX vx_nr1, FIDX *dofs,
		       struct mesh *m
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Poison equation with linear
   triangle elements (T1) on the (T2) mesh m

   Input:  m       - the mesh
           vx_nr1  - the number of DOFs, (the number of corner nodes of
                     the elements in the mesh)
	   dofs    - pointer to array of length m.vx_nr, such that 
	             dofs[i] tells which (if any) degree of freedom
		     (DOF) belongs to the node, 
		     dofs[i]=j ==> j-th DOF belongs to the i-th node
		     of the mesh

   Output: K       - stiffness matrix

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_lame_tx(struct sparse *K, struct projector1 *P, struct mesh *msh, struct vector *rhs, struct vector *u0,
                  double lambda, double mu,
		  int gammas, int bcswitch, int type
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Lame' equation (linear
   elasticity) on the mesh msh, such that 
       K u = 0
   defines the approximate solution u of the Lame' equation
       mu*Laplace(u) + (mu+lambda)*grad div(u) = 0
   with boundary conditions as given in the mesh msh
   (p1 or p2 triangles for now)

   Input:  msh     - the mesh
           lambda  - Lame' constant 
           mu      - Lame' constant 
           gammas  - the stiffening parameter, usefull when this is
                     used for mesh deformation, then gammas=0,
		     FOR REAL LINEAR ELASTICITY USE gammas=1 !
           bcswitch- if bcswitch==0 only specified Dirichlet BC are used
                     if ==1 ALL types of boundary conditions
                     are treated as Dirichlet, ignoring their
                     specified type
                     if ==2 Dirichlet BC and Neumann BC are used
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: K       - stiffness matrix
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC, if not wanted
           	     give P=NULL

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_neum_bc_t2( struct mesh *m, struct vector *rhs
/* adds the contribution of the Neumann boundary conditions to the rhs
       sigma(u)*n=g on Gamma_N
   ==>
       rhs(v) += int_{Gamma_N} +g*v ds

   Input:  m       - the mesh (2d T2), input

   Output: rhs     - rhighthandside vector, set according to the
                     integral formula above

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_elem_detA_tx(struct mesh *m, double *elem_detA, int type
/* computes the detA the determinate of the element linear mapping 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: elem_detA-detA for each element

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int max_angle_tx(struct mesh *m, double *maxal, int type
/* computes the maximum interior angle amon the triangular elements in
   the mesh 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: maxal   - maximum interior angle for all elements, in
                     radiant (rather than degrees), given by reference

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
