/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE assembly.c
HEADER assembly.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"
#include "sparse_struct.h"

*/

#include <math.h>


/* function prototypes */
#include "assembly.h"

/* prototypes of external functions */
#include "cubature.h"
#include "sparse.h"
#include "mesh.h"
#include "feins_lapack.h"    /* link against LAPACK! */

/*FUNCTION*/
int assem_poison_t1(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P, 
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poison equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poison equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (only p1 triangles for now)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   u0      - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el,bd,i,j,k,r;
  int err;

  FIDX dim=2, bas_n, vx_nr, el_w, vx_w, eg_w, fc_w, bd_w, fu_w;
  FIDX subtypes[1];

  /* for FORTRAN calls: */
  char fTrans='T', fNoTrans='N';
  double dhelp, done=1.0, dzero=0.0;
  int fdim, fbas_n;

  struct int_data iform; /* integration formula    */
  double *elK;           /* element stiffnes matrx */
  double *Jac, *Jacinv;  /* Jacobian of the element mapping and its
			    inverse */
  double x_k[2],fx_k;    /* world coordinates of the quadrature point
			    and a function value at this point */
  double *Jinvgrad;      /* inverse Jacobian times gradphi */
  double detJac;         /* determinant of the Jacobian */
  double *phi, *gradp;

  /* init */
  /* get integration formula */
  subtypes[0]=1;

  err=cubature_bases( dim, 1, tria, 1, subtypes, &iform); 

  if (err!=SUCCESS)
    {
      fprintf( stderr,
	       "cubature_bases returned error in assem_poison_t1!\n");
      return FAIL;
    }

  /* make phi and gradphi better accessible */
  phi   = (iform.bases[0]->phi);
  gradp = (iform.bases[0]->gradphi);
  bas_n = (iform.bases[0]->num_basis);

  fdim  = (int) dim;
  fbas_n= (int) bas_n;

  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;
  eg_w  = (*m).eg_w;
  fc_w  = (*m).fc_w;
  bd_w  = (*m).bd_w;
  fu_w  = (*m).fu_w;

  /* allocate memory for elK */
  TRY_MALLOC( elK, bas_n*bas_n, double, assem_poison_t1);
  /* allocate memory for Jac, Jacinf */
  TRY_MALLOC( Jac, dim*dim, double, assem_poison_t1);
  TRY_MALLOC( Jacinv, dim*dim, double, assem_poison_t1);
  /* allocate memory for Jinvgrad */
  TRY_MALLOC( Jinvgrad, dim*bas_n, double, assem_poison_t1);

  /* clear K */
  sparse_empty(K);
  /* clear rhs, u0, P */
  for (i=0; i<vx_nr; i++)
    {
      (*rhs).V[i]=0.0;
      (*u0).V[i] =0.0;
      (*P).V[i]=0;
    }

  /* loop over all elements */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* set elK to zero */
      for (i=0; i<bas_n*bas_n; i++) 
	{
	  elK[i]=0.0;
	}
      
      /* loop over all integration points */
      for (k=0; k<iform.num_points; k++)
	{
	  /* compute x_k and the Jacobian at this point */
	  /* x_k=0 */
	  for (i=0; i<dim; i++)
	    x_k[i]=0.0;
	  /* Jac=0 */
	  for (i=0; i<dim*dim; i++)
	    Jac[i]=0.0;
	  
	  /* x_k = sum_{i=nodes} vertex(i)*phi_i,
	     Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
	  for (i=0;i<bas_n; i++)
	    {
	      for (j=0; j<dim; j++)
		{
		  x_k[j]+= 
		    (*m).vertex[(*m).elem[el*el_w+MCT1ELNOD1+i]*vx_w
				+MCT1VXSTRT+j]
		    * phi[k*bas_n+i];
		  for (r=0; r<dim; r++)
		    {
		      Jac[j*dim+r]+= 
			(*m).vertex[(*m).elem[el*el_w+MCT1ELNOD1+i]*vx_w
				    +MCT1VXSTRT+j]
			* gradp[k*bas_n*dim +i*dim +r];
		    }
		}
	    }
	  /* print Jac: 
	  printf("\nelement %d Jac:\n",(int) el); for (i=0; i<dim; i++) {
	  printf("%7f %7f \n", Jac[i*dim+0], Jac[i*dim+1]); } */

	  /* get detJac */
	  detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];
	  
	  /* get Jacinv (here direct) */
	  Jacinv[0]=1.0/detJac*Jac[3];
	  Jacinv[1]=-1.0/detJac*Jac[1];
	  Jacinv[2]=-1.0/detJac*Jac[2];
	  Jacinv[3]=1.0/detJac*Jac[0];

	  /* Jinvgrad= Jacinv * gradphi[k,:,:]  */
	  dgemm_( &fNoTrans, &fNoTrans, &fdim, &fbas_n, &fdim, 
		  &done, Jacinv, &fdim, &(gradp[k*bas_n*dim]), &fdim,
		  &dzero, Jinvgrad, &fdim );

	  
	  dhelp=iform.weights[k]*fabs(detJac);
	  /* elK += |detJac|*weigth[k] * Jinvgrad^T*Jinvgrad */
	  dgemm_( &fTrans, &fNoTrans, &fbas_n, &fbas_n, &fdim,
		  &dhelp, Jinvgrad, &fdim, Jinvgrad, &fdim,
		  &done, elK, &fbas_n );

	  /* acumulate rhs */
	  if ((*m).face[el*fc_w+MCT1FCRHSF]>=0)
	    {
	      /* rhs_i = int (f*phi_i) 
             	       = sum weight_k*|detJac| * f(x_k) * phi(x_k) */
	      /* get f(x_k) */
	      err=mesh_func_eval ( m, (*m).face[el*fc_w+MCT1FCRHSF],
				   x_k, 1, &fx_k);
	      FUNCTION_FAILURE_HANDLE(err,mesh_func_eval,
				      assem_poison_t1);
	      for (i=0; i<bas_n; i++)
		(*rhs).V[(*m).elem[el*el_w+i]]+=
		  iform.weights[k]*fabs(detJac)*fx_k*phi[k*bas_n+i];
	    }
	} /* end loop over all integration points */
      /* print elK:
	 printf("\nelement %d:\n",(int) el); for (i=0; i<bas_n; i++) {
	 printf("%3d %7f %7f %7f\n",(int)(*m).elem[el*el_w+MCT1ELNOD1+i], elK[i*bas_n+0],
	 elK[i*bas_n+1], elK[i*bas_n+2]); } /* */
      
      /* elK is ready, add to K */
      err=sparse_add_local(K, NoTrans,
			   bas_n, &((*m).elem[el*el_w+MCT1ELNOD1]),
			   bas_n, &((*m).elem[el*el_w+MCT1ELNOD1]),
			   elK, bas_n );
      FUNCTION_FAILURE_HANDLE( err, sparse_add_local, assem_poison_t1);
	       
    }
  /* end loop over all elements */

#warning poison_T1: Neumann BC not accounted for in rhs yet!
  /* first Neumann BC, otherwise they would collide with Dirichlet BC
  */


  /* loop over boundary, Dirichlet BC implementation */
  for (bd=0; bd<(*m).bd_nr; bd++)
    {
      /* take Dirichlet BC into account */
      if ((*m).bound[bd*bd_w + MCT1BDTYPE]==1)
	{
	  FIDX edi, node1, node2, fu, err1, err2;
	  
	  /* check out which nodes */
	  edi   = (*m).bound[bd*bd_w + MCT1BDEDGE];
	  node1 = (*m).edge[edi*eg_w + MCT1EGNOD1];
	  node2 = (*m).edge[edi*eg_w + MCT1EGNOD1 +1];

	  fu=(*m).bound[bd*bd_w + MCT1BDFNCT];

	  err1=mesh_func_eval(m, fu,
			      &(*m).vertex[node1*vx_w+MCT1VXSTRT],
			      1, &(*u0).V[node1]);
	  err2=mesh_func_eval(m, fu,
			      &(*m).vertex[node2*vx_w+MCT1VXSTRT],
			      1, &(*u0).V[node2]);
	  if ((err1!=SUCCESS)||(err2)!=SUCCESS)
	    {
	      fprintf(stderr,
		      "assem_poison_t1: function_eval error in DiriBC\n");
	      return FAIL;
	    }


	  /* at this stage the projector array holds the info if a DOF
	     is Dirichlet
	  */
	  (*P).V[node1]=1;
	  (*P).V[node2]=1;

	} /* end this BD is Dirichlet */
      
    } /* end take Dirichlet BC into account */
  /* now correct the projector to a list of Dirichlet DOFs */
  j=0;
  for (i=0; i<vx_nr; i++)
    {
      if ((*P).V[i]==1)
	{
	  /* this is a Dirichlet node, append it to the list */
	  (*P).V[j]=i;
	  j++;
	}
    }
  (*P).len=j;

  if ( (*K).type == SP_TYPE_FLEX )
    {
      err = sparse_convert_compressed_row( K );
      FUNCTION_FAILURE_HANDLE( err, sparse_convert_compressed_row, 
			       assem_poison_t1 );
    }

  /* free local data */
  free_intdata (&iform);
  free(elK);
  free(Jac);
  free(Jacinv);
  free(Jinvgrad);

  return SUCCESS;
}
  


/*FUNCTION*/
int assem_poison_t2(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P,
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poison equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poison equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   u0      - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el,bd,i,j,k,r;
  int err;

  FIDX dim=2, bas_n, vx_nr, el_w, vx_w, eg_w, fc_w, bd_w, fu_w;
  FIDX subtypes[1];

  /* for FORTRAN calls: */
  char fTrans='T', fNoTrans='N';
  double dhelp, done=1.0, dzero=0.0;
  int fdim, fbas_n;

  struct int_data iform; /* integration formula    */
  double *elK;           /* element stiffnes matrx */
  double *Jac, *Jacinv;  /* Jacobian of the element mapping and its
			    inverse */
  double x_k[2],fx_k;    /* world coordinates of the quadrature point
			    and a function value at this point */
  double *Jinvgrad;      /* inverse Jacobian times gradphi */
  double detJac;         /* determinant of the Jacobian */
  double *phi, *gradp;

  /* init */
  /* get integration formula */
  subtypes[0]=2;

  err=cubature_bases( dim, 4, tria, 1, subtypes, &iform); 

  if (err!=SUCCESS)
    {
      fprintf( stderr,
	       "cubature_bases returned error in assem_poison_t1!\n");
      return FAIL;
    }

  /* make phi and gradphi better accessible */
  phi   = (iform.bases[0]->phi);
  gradp = (iform.bases[0]->gradphi);
  bas_n = (iform.bases[0]->num_basis);

  fdim  = (int) dim;
  fbas_n= (int) bas_n;

  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;
  eg_w  = (*m).eg_w;
  fc_w  = (*m).fc_w;
  bd_w  = (*m).bd_w;
  fu_w  = (*m).fu_w;

  /* allocate memory for elK */
  TRY_MALLOC( elK, bas_n*bas_n, double, assem_poison_t1);
  /* allocate memory for Jac, Jacinf */
  TRY_MALLOC( Jac, dim*dim, double, assem_poison_t1);
  TRY_MALLOC( Jacinv, dim*dim, double, assem_poison_t1);
  /* allocate memory for Jinvgrad */
  TRY_MALLOC( Jinvgrad, dim*bas_n, double, assem_poison_t1);

  /* clear K */
  sparse_empty(K);
  /* clear rhs, u0, P */
  for (i=0; i<vx_nr; i++)
    {
      (*rhs).V[i]=0.0;
      (*u0).V[i] =0.0;
      (*P).V[i]=0;
    }

  /* loop over all elements */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* set elK to zero */
      for (i=0; i<bas_n*bas_n; i++) 
	{
	  elK[i]=0.0;
	}
      
      /* loop over all integration points */
      for (k=0; k<iform.num_points; k++)
	{
	  /* compute x_k and the Jacobian at this point */
	  /* x_k=0 */
	  for (i=0; i<dim; i++)
	    x_k[i]=0.0;
	  /* Jac=0 */
	  for (i=0; i<dim*dim; i++)
	    Jac[i]=0.0;
	  
	  /* x_k = sum_{i=nodes} vertex(i)*phi_i,
	     Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
	  for (i=0;i<bas_n; i++)
	    {
	      for (j=0; j<dim; j++)
		{
		  x_k[j]+= 
		    (*m).vertex[(*m).elem[el*el_w+MCT2ELNOD1+i]*vx_w
				+MCT2VXSTRT+j]
		    * phi[k*bas_n+i];
		  for (r=0; r<dim; r++)
		    {
		      Jac[j*dim+r]+= 
			(*m).vertex[(*m).elem[el*el_w+MCT2ELNOD1+i]*vx_w
				    +MCT2VXSTRT+j]
		      * gradp[k*bas_n*dim +i*dim +r];
		    }
		}
	    }
	  /* print Jac: 
	  printf("\nelement %d Jac:\n",(int) el); for (i=0; i<dim; i++) {
	  printf("%7f %7f \n", Jac[i*dim+0], Jac[i*dim+1]); } */

	  /* get detJac */
	  detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];
	  
	  /* printf("el=%3d, k=%2d, detJac= %f\n", (int) el, (int) k,
	     detJac); /* */
	  
	  /* get Jacinv (here direct) */
	  Jacinv[0]=1.0/detJac*Jac[3];
	  Jacinv[1]=-1.0/detJac*Jac[1];
	  Jacinv[2]=-1.0/detJac*Jac[2];
	  Jacinv[3]=1.0/detJac*Jac[0];

	  /* Jinvgrad= Jacinv * gradphi[k,:,:]  */
	  dgemm_( &fNoTrans, &fNoTrans, &fdim, &fbas_n, &fdim,
		  &done, Jacinv, &fdim, &(gradp[k*bas_n*dim]), &fdim ,
		  &dzero, Jinvgrad, &fdim );

	  
	  dhelp=iform.weights[k]*fabs(detJac);
	  /* elK += |detJac|*weigth[k] * Jinvgrad^T*Jinvgrad */
	  dgemm_( &fTrans, &fNoTrans, &fbas_n, &fbas_n, &fdim,
		  &dhelp, Jinvgrad, &fdim, Jinvgrad, &fdim,
		  &done, elK, &fbas_n );

	  /* acumulate rhs */
	  if ((*m).face[el*fc_w+MCT2FCRHSF]>=0)
	    {
	      /* rhs_i = int (f*phi_i) 
             	       = sum weight_k*|detJac| * f(x_k) * phi(x_k) */
	      /* get f(x_k) */
	      err=mesh_func_eval ( m, (*m).face[el*fc_w+MCT2FCRHSF],
				   x_k, 1, &fx_k);
	      FUNCTION_FAILURE_HANDLE(err,mesh_func_eval,
				      assem_poison_t2);
	      for (i=0; i<bas_n; i++)
		(*rhs).V[(*m).elem[el*el_w+i]]+=
		  iform.weights[k]*fabs(detJac)*fx_k*phi[k*bas_n+i];
	    }
	} /* end loop over all integration points */
      /* print elK: 
	 printf("\nelement %d:\n", (int) el); for (i=0; i<bas_n; i++) {
	 printf("%3d %4f %4f %4f %4f %4f %4f\n",
	 (int) (*m).elem[el*el_w+MCT2ELNOD1+i],
	 elK[i*bas_n+0], elK[i*bas_n+1], elK[i*bas_n+2], 
	 elK[i*bas_n+3], elK[i*bas_n+4], elK[i*bas_n+5]); } /* */
      
      /* elK is ready, add to K */
      err=sparse_add_local(K, NoTrans,
			   bas_n, &((*m).elem[el*el_w+MCT2ELNOD1]),
			   bas_n, &((*m).elem[el*el_w+MCT2ELNOD1]),
			   elK, bas_n );
      FUNCTION_FAILURE_HANDLE( err, sparse_add_local, assem_poison_t2);
		       
    }
  /* end loop over all elements */

#warning poison_T2: Neumann BC not accounted for in rhs yet!
  /* first Neumann BC, otherwise they would collide with Dirichlet BC
  */


  /* loop over boundary, Dirichlet BC implementation */
  for (bd=0; bd<(*m).bd_nr; bd++)
    {
      /* take Dirichlet BC into account */
      if ((*m).bound[bd*bd_w + MCT2BDTYPE]==1)
	{
	  FIDX edi, node1, node2, nodem, fu, err1, err2, errm;
	  
	  /* check out which nodes */
	  edi   = (*m).bound[bd*bd_w + MCT2BDEDGE];
	  node1 = (*m).edge[edi*eg_w + MCT2EGNOD1];
	  node2 = (*m).edge[edi*eg_w + MCT2EGNOD1 +1];
	  nodem = (*m).edge[edi*eg_w + MCT2EGNODM];

	  fu=(*m).bound[bd*bd_w + MCT2BDFNCT];

	  err1=mesh_func_eval(m, fu,
			      &(*m).vertex[node1*vx_w+MCT2VXSTRT],
			      1, &(*u0).V[node1]);
	  err2=mesh_func_eval(m, fu,
			      &(*m).vertex[node2*vx_w+MCT2VXSTRT],
			      1, &(*u0).V[node2]);
	  errm=mesh_func_eval(m, fu,
			      &(*m).vertex[nodem*vx_w+MCT2VXSTRT],
			      1, &(*u0).V[nodem]);
	  if (((err1!=SUCCESS)||(err2)!=SUCCESS)||((errm)!=SUCCESS))
	    {
	      fprintf(stderr,
		      "assem_poison_t2: function_eval error in DiriBC\n");
	      return FAIL;
	    }


	  /* at this stage the projector array holds the info if a DOF
	     is Dirichlet
	  */
	  (*P).V[node1]=1;
	  (*P).V[node2]=1;
	  (*P).V[nodem]=1;

	} /* end this BD is Dirichlet */
      
    } /* end take Dirichlet BC into account */
  /* now correct the projector to a list of Dirichlet DOFs */
  j=0;
  for (i=0; i<vx_nr; i++)
    {
      if ((*P).V[i]==1)
	{
	  /* this is a Dirichlet node, append it to the list */
	  (*P).V[j]=i;
	  j++;
	}
    }
  (*P).len=j;

  if ( (*K).type == SP_TYPE_FLEX )
    {
      err = sparse_convert_compressed_row( K );
      FUNCTION_FAILURE_HANDLE( err, sparse_convert_compressed_row, 
			       assem_poison_t2 );
    }

  /* free local data */
  free_intdata (&iform);
  free(elK);
  free(Jac);
  free(Jacinv);
  free(Jinvgrad);

  return SUCCESS;
}
  
  
  
/*FUNCTION*/
int assem_poison_t1_t2(struct sparse *K, FIDX vx_nr1, FIDX *dofs,
		       struct mesh *m
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Poison equation with linear
   triangle elements (T1) on the (T2) mesh m

   Input:  m       - the mesh
           vx_nr1  - the number of DOFs, (the number of corner nodes of
                     the elements in the mesh)
	   dofs    - pointer to array of length m.vx_nr, such that 
	             dofs[i] tells which (if any) degree of freedom
		     (DOF) belongs to the node, 
		     dofs[i]=j ==> j-th DOF belongs to the i-th node
		     of the mesh

   Output: K       - stiffness matrix

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el, i, j, k, r;
  int err;

  FIDX dim=2, bas_n, vx_nr, el_w, vx_w;
  FIDX subtypes[1];

  /* for FORTRAN calls: */
  char fTrans='T', fNoTrans='N';
  double dhelp, done=1.0, dzero=0.0;
  int fdim, fbas_n;

  struct int_data iform; /* integration formula    */
  double *elK;           /* element stiffnes matrx */
  FIDX *elDOFS;           /* the DOFs belonging to the nodes of the
			    element */
  double *Jac, *Jacinv;  /* Jacobian of the element mapping and its
			    inverse */
  double *Jinvgrad;      /* inverse Jacobian times gradphi */
  double detJac;         /* determinant of the Jacobian */
  double *phi, *gradp;

  /* init */
  /* get integration formula */
  subtypes[0]=1;

  err=cubature_bases( dim, 1, tria, 1, subtypes, &iform); 

  if (err!=SUCCESS)
    {
      fprintf( stderr,
	       "assem_poison_t1_t2: cubature_bases returned error!\n");
      return FAIL;
    }

  /* make phi and gradphi better accessible */
  phi   = (iform.bases[0]->phi);
  gradp = (iform.bases[0]->gradphi);
  bas_n = (iform.bases[0]->num_basis);

  fdim  = (int) dim;
  fbas_n= (int) bas_n;

  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;

  /* allocate memory for elK */
  TRY_MALLOC( elK, bas_n*bas_n, double, assem_poison_t1_t2);
  /* allocate memory for elDOFS */
  TRY_MALLOC( elDOFS, bas_n, FIDX, assem_poison_t1_t2);
  /* allocate memory for Jac, Jacinf */
  TRY_MALLOC( Jac, dim*dim, double, assem_poison_t1_t2);
  TRY_MALLOC( Jacinv, dim*dim, double, assem_poison_t1_t2);
  /* allocate memory for Jinvgrad */
  TRY_MALLOC( Jinvgrad, dim*bas_n, double, assem_poison_t1_t2);

  /* clear K */
  sparse_empty(K);

  /* loop over all elements */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* set elK to zero */
      for (i=0; i<bas_n*bas_n; i++) 
	{
	  elK[i]=0.0;
	}
      
      /* loop over all integration points */
      for (k=0; k<iform.num_points; k++)
	{
	  /* compute the Jacobian at this point */
	  /* Jac=0 */
	  for (i=0; i<dim*dim; i++)
	    Jac[i]=0.0;
	  
	  /* Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
	  for (i=0;i<bas_n; i++)
	    {
	      for (j=0; j<dim; j++)
		{
		  for (r=0; r<dim; r++)
		    {
		      Jac[j*dim+r]+= 
			(*m).vertex[(*m).elem[el*el_w+MCT2ELNOD1+i]*vx_w
				    +MCT2VXSTRT+j]
			* gradp[k*bas_n*dim +i*dim +r];
		    }
		}
	    }

	  /* get detJac */
	  detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];
	  
	  /* get Jacinv (here direct) */
	  Jacinv[0]=1.0/detJac*Jac[3];
	  Jacinv[1]=-1.0/detJac*Jac[1];
	  Jacinv[2]=-1.0/detJac*Jac[2];
	  Jacinv[3]=1.0/detJac*Jac[0];

	  /* Jinvgrad= Jacinv * gradphi[k,:,:]  */
	  dgemm_( &fNoTrans, &fNoTrans, &fdim, &fbas_n, &fdim,
		  &done, Jacinv, &fdim, &(gradp[k*bas_n*dim]), &fdim,
		  &dzero, Jinvgrad, &fdim );

	  dhelp=iform.weights[k]*fabs(detJac);
	  /* elK += |detJac|*weigth[k] * Jinvgrad^T*Jinvgrad */
	  dgemm_( &fTrans, &fNoTrans, &fbas_n, &fbas_n, &fdim,
		  &dhelp, Jinvgrad, &fdim, Jinvgrad, &fdim,
		  &done, elK, &fbas_n );

	} /* end loop over all integration points */

      for (i=0; i<bas_n; i++)
	{
	  elDOFS[i]=dofs[(*m).elem[el*el_w+MCT2ELNOD1+i]];
	  if (elDOFS[i]<0)
	    {
	      fprintf(stderr,
		      "assem_poison_t1_t2: dofs don't match mesh!\n");
	      printf("elDOFS[%d]=%d\n", (int) i, (int) elDOFS[i]);
	      return FAIL;
	    }
	}

      /* elK is ready, add to K */
      err=sparse_add_local(K, NoTrans, bas_n, elDOFS, bas_n, elDOFS,
			   elK, bas_n );
      FUNCTION_FAILURE_HANDLE( err, sparse_add_local, assem_poison_t1_t2);
	       
    }

  if ( (*K).type == SP_TYPE_FLEX )
    {
      err = sparse_convert_compressed_row( K );
      FUNCTION_FAILURE_HANDLE( err, sparse_convert_compressed_row, 
			       assem_poison_t1_t2 );
    }

  /* end loop over all elements */
  /* free local data */
  free_intdata (&iform);
  free(elK);
  free(elDOFS);
  free(Jac);
  free(Jacinv);
  free(Jinvgrad);

  return SUCCESS;
}



/*******************************************************************
 *******************************************************************
 *                                                                 *
 *        Lame'                                                    *
 *                                                                 *
 *******************************************************************
 *******************************************************************/




/*FUNCTION*/
int assem_lame_tx(struct sparse *K, struct projector1 *P, struct mesh *msh, struct vector *rhs, struct vector *u0,
                  double lambda, double mu,
		  int gammas, int bcswitch, int type
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Lame' equation (linear
   elasticity) on the mesh msh, such that 
       K u = 0
   defines the approximate solution u of the Lame' equation
       mu*Laplace(u) + (mu+lambda)*grad div(u) = 0
   with boundary conditions as given in the mesh msh
   (p1 or p2 triangles for now)

   Input:  msh     - the mesh
           lambda  - Lame' constant 
           mu      - Lame' constant 
           gammas  - the stiffening parameter, usefull when this is
                     used for mesh deformation, then gammas=0,
		     FOR REAL LINEAR ELASTICITY USE gammas=1 !
           bcswitch- if bcswitch==0 only specified Dirichlet BC are used
                     if ==1 ALL types of boundary conditions
                     are treated as Dirichlet, ignoring their
                     specified type
                     if ==2 Dirichlet BC and Neumann BC are used
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: K       - stiffness matrix
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC, if not wanted
           	     give P=NULL

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el,bd,i,j,k,r,s,m,n;
  int err;

  FIDX dim=2, bas_n, dim_bas_n, bas_n1, 
    vx_nr, el_w, vx_w, eg_w, fc_w, bd_w, fu_w;
  FIDX subtypes[2];

  /* for FORTRAN calls: */
  char   fNoTrans='N';
  double dhelp, done=1.0, dzero=0.0;
  int fdim, fbas_n, fbas_n1;

  struct int_data iform; /* integration formula    */
  double *elK;           /* element stiffnes matrx */
  double *Jac, *Jacinv;  /* Jacobian of the element mapping and its
			    inverse */
  FIDX    *ldofs;         /* list of local dofs */
  double *Jinvgrad;      /* inverse Jacobian times gradphi */
  double detJac;         /* determinant of the Jacobian */
  double AdetJac;        /* abs(detJac) */
  double *phi, *gradp;
  double *phi1, *gradp1;

  FIDX l_mctxELNOD1, l_mctxVXSTRT, l_mctxBDTYPE, l_mctxBDEDGE, l_mctxEGNOD1;

  switch (type)
    {
    case 1:
      l_mctxVXSTRT = MCT1VXSTRT;
      l_mctxEGNOD1 = MCT1EGNOD1;
      l_mctxELNOD1 = MCT1ELNOD1;
      l_mctxBDTYPE = MCT1BDTYPE;
      l_mctxBDEDGE = MCT1BDEDGE;
      break;
    case 2:
      l_mctxVXSTRT = MCT2VXSTRT;
      l_mctxEGNOD1 = MCT2EGNOD1;
      l_mctxELNOD1 = MCT2ELNOD1;
      l_mctxBDTYPE = MCT2BDTYPE;
      l_mctxBDEDGE = MCT2BDEDGE;
      break;
    default:
      fprintf(stderr,"assem_lame_tx: unknown type=%d\n",type);
      return FAIL;
    }

  /* init */
  /* get integration formula */
  subtypes[0]=1;
  subtypes[1]=type;

  err=cubature_bases( dim, 2, tria, 2, subtypes, &iform); 
  if (err!=SUCCESS)
    {
      fprintf( stderr,
	       "cubature_bases returned error in assem_lame_tx!\n");
      return FAIL;
    }

  /* make phi and gradphi better accessible */
  phi1  = (iform.bases[0]->phi);
  gradp1= (iform.bases[0]->gradphi);
  bas_n1= (iform.bases[0]->num_basis);
  phi   = (iform.bases[1]->phi);
  gradp = (iform.bases[1]->gradphi);
  bas_n = (iform.bases[1]->num_basis);

  fdim    = (int) dim;
  fbas_n  = (int) bas_n;
  fbas_n1 = (int) bas_n1;

  vx_nr = (*msh).vx_nr;
  vx_w  = (*msh).vx_w;
  el_w  = (*msh).el_w;
  eg_w  = (*msh).eg_w;
  fc_w  = (*msh).fc_w;
  bd_w  = (*msh).bd_w;
  fu_w  = (*msh).fu_w;

  dim_bas_n = dim*bas_n;

  /* allocate memory for elK */
  TRY_MALLOC( elK, dim_bas_n*dim_bas_n, double, assem_lame_tx);
  /* allocate memory for Jac, Jacinf */
  TRY_MALLOC( Jac, dim*dim, double, assem_lame_tx);
  TRY_MALLOC( Jacinv, dim*dim, double, assem_lame_tx);
  /* allocate memory for Jinvgrad */
  TRY_MALLOC( Jinvgrad, dim*bas_n, double, assem_lame_tx);
  /* allocate memory for ldofs */
  TRY_MALLOC( ldofs, dim_bas_n, FIDX, assem_lame_tx);

  /* clear K */
  sparse_empty(K);

  if (P!=NULL)
    {
      /* clear P */
      for (i=0; i<dim*vx_nr; i++)
	{
	  (*P).V[i]=0;
	}
    }


  /* loop over all elements */
  for (el=0; el<(*msh).el_nr; el++)
    {
      /* set elK to zero */
      for (i=0; i < dim_bas_n*dim_bas_n; i++) 
	{
	  elK[i]=0.0;
	}
      
      /* compute the Jacobian for this element 
	 (constant on the element)*/
      /* Jac=0 */
      for (i=0; i<dim*dim; i++)
	Jac[i]=0.0;
	  
      /* Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
      for (i=0;i<bas_n1; i++)
	{
	  for (j=0; j<dim; j++)
	    for (r=0; r<dim; r++)
	      {
		Jac[j*dim+r]+= 
		  (*msh).vertex[(*msh).elem[el*el_w+l_mctxELNOD1+i]*vx_w
			      +l_mctxVXSTRT+j]
		  * gradp1[0*bas_n1*dim +i*dim +r];
	      }
	}

      /* define ldofs */
      for (i=0;i<bas_n; i++)
	{
	  for (j=0; j<dim; j++)
	    {
	      ldofs[j*bas_n+i]=(*msh).elem[el*el_w+l_mctxELNOD1+i]+j*vx_nr;
	    }
	}
      /* print Jac: 
	 printf("\nelement %d Jac:\n", (int) el); for (i=0; i<dim; i++) {
	 printf("%7f %7f \n", Jac[i*dim+0], Jac[i*dim+1]); } */

      /* get detJac */
      detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];
	  
      /* get Jacinv (here direct) */
      Jacinv[0]=1.0/detJac*Jac[3];
      Jacinv[1]=-1.0/detJac*Jac[1];
      Jacinv[2]=-1.0/detJac*Jac[2];
      Jacinv[3]=1.0/detJac*Jac[0];

      if (gammas==0)
	AdetJac = 1.0;
      else
	AdetJac = fabs(detJac);

      /* loop over all integration points */
      for (k=0; k<iform.num_points; k++)
	{
	  /* real world gradient, not constant for quadratic elements */
	  /* Jinvgrad= Jacinv * gradphi[k,:,:]  */
	  dgemm_( &fNoTrans, &fNoTrans, &fdim, &fbas_n, &fdim, 
		  &done, Jacinv, &fdim, &(gradp[k*bas_n*dim]), &fdim,
		  &dzero, Jinvgrad, &fdim );

	  /* #define WE_TEST_SOMETHING */
#ifndef WE_TEST_SOMETHING
	  /* the lambda*div(u)*div(v) part */
	  dhelp=iform.weights[k]*AdetJac*lambda;
	  for (n=0; n<dim; n++) /* dim v */
	    for (m=0; m<dim; m++) /* dim u */
	      for (i=0; i<bas_n; i++) /* dof v */
		for (j=0; j<bas_n; j++) /* dof u */
		  {
		    elK[ (m*bas_n+j) * dim_bas_n + (n*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+m]*Jinvgrad[i*dim+n];
			     
		  }

	  /* the 2*mu*eps:eps part */
	  dhelp=iform.weights[k]*AdetJac*0.5*mu;
	  for (r=0; r<dim; r++) 
	    for (s=0; s<dim; s++)
	      for (i=0; i<bas_n; i++) /* dof v */
		for (j=0; j<bas_n; j++) /* dof u */
		  {
		    elK[ (r*bas_n+j) * dim_bas_n + (r*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+s]*Jinvgrad[i*dim+s];
		    elK[ (r*bas_n+j) * dim_bas_n + (s*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+s]*Jinvgrad[i*dim+r];
		    elK[ (s*bas_n+j) * dim_bas_n + (r*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+r]*Jinvgrad[i*dim+s];
		    elK[ (s*bas_n+j) * dim_bas_n + (s*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+r]*Jinvgrad[i*dim+r];
		  } 
#else
#warning "Lame replaced by Laplacian for tests?"

	  /* grad(u)*grad(v) test (Laplacian) */
	  dhelp=iform.weights[k]*AdetJac;
	  for (r=0; r<dim; r++) /* dim dot */
	    for (s=0; s<dim; s++) /* dim u,v */
	      for (i=0; i<bas_n; i++) /* dof v */
		for (j=0; j<bas_n; j++) /* dof u */
		  {
		    elK[ (s*bas_n+j) * dim_bas_n + (s*bas_n+i)] +=
		      dhelp* Jinvgrad[j*dim+r]*Jinvgrad[i*dim+r];
			     
		  }
#endif


	} /* end loop over all integration points */
      /* print elK:
	 printf("\nelement %d:\n", (int) el); for (i=0; i<bas_n; i++) {
	 printf("%3d %7f %7f %7f\n",
	 (int) (*msh).elem[el*el_w+l_mctxELNOD1+i], elK[i*bas_n+0],
	 elK[i*bas_n+1], elK[i*bas_n+2]); } /* */
      
      /* elK is ready, add to K */
      err=sparse_add_local(K, NoTrans,
			   dim_bas_n, ldofs, dim_bas_n, ldofs,
			   elK, dim_bas_n );
      FUNCTION_FAILURE_HANDLE( err, sparse_add_local, assem_lame_tx);
	       
    }
  /* end loop over all elements */

  if (P!=NULL) 
    {
    if ((bcswitch==0)||(bcswitch==1))
      {
        /* loop over boundary, Dirichlet BC implementation */
        for (bd=0; bd<(*msh).bd_nr; bd++)
	  {
	    /* take Dirichlet BC into account */
	    if ( ((*msh).bound[bd*bd_w + l_mctxBDTYPE]==1) || (bcswitch==1) )
	      {
	        FIDX edi, node1, node2;
	      
	        /* check out which nodes */
	        edi   = (*msh).bound[bd*bd_w + l_mctxBDEDGE];
	        node1 = (*msh).edge[edi*eg_w + l_mctxEGNOD1];
	        node2 = (*msh).edge[edi*eg_w + l_mctxEGNOD1 +1];
 
	        /* at this stage the projector array holds the info if a DOF
		   is Dirichlet */
	        (*P).V[node1]=1;
	        (*P).V[node2]=1;
	      
	        if (type==2)
		  {
		    FIDX nodem;
		    nodem = (*msh).edge[edi*eg_w + MCT2EGNODM];
		    (*P).V[nodem]=1;
		  }

	      } /* end this BD is Dirichlet */
      
	  } /* end take Dirichlet BC into account */
        /* now correct the projector to a list of Dirichlet DOFs */
        j=0;
        for (i=0; i<vx_nr; i++)
  	  {
	    if ((*P).V[i]==1)
	      {
	        /* this is a Dirichlet node, append it to the list */
	        (*P).V[j]=i;
	        j++;
	      }
	  }
        /* first dimenstion is set, set the others too */
        r=j;
        for (s=1; s<dim; s++)
	  for (i=0; i<r; i++)
	    {
	      (*P).V[i+s*r]=(*P).V[i]+s*vx_nr;
	    }
        (*P).len=dim*r;
  
        printf("vx_nr=%d   P.len=%d\n", (int) vx_nr, (int) (*P).len);
      } /* end define P */

    else /*Dirichlet BC and Neumann BC */
      {
        for (i=0; i<(*msh).bd_nr; i++)
          {

            /* take Dirichlet BC into account */
            if ((*msh).bound[i*(*msh).bd_w + MCT2BDTYPE]==1)
	      {
	        FIDX edi, node1, node2, node3, fu, err;

                double *feval_result;
                TRY_MALLOC( feval_result, dim, double, assem_lame_tx);

	        /* check out which nodes */
	        edi   = (*msh).bound[  i*(*msh).bd_w + MCT2BDEDGE];
	        node1 = (*msh).edge [edi*(*msh).eg_w + MCT2EGNOD1];
	        node2 = (*msh).edge [edi*(*msh).eg_w + MCT2EGNOD1 +1];
	        node3 = (*msh).edge [edi*(*msh).eg_w + MCT2EGNODM];

	        fu=(*msh).bound[i*(*msh).bd_w + MCT2BDFNCT];

	        /* NODE 1 Fct-eval and copying to nodem.V */
	        err=mesh_func_eval(&(*msh), fu,
			           &(*msh).vertex[node1*(*msh).vx_w+MCT2VXSTRT],
			           dim, feval_result);

	        if ((err)!=SUCCESS)
	          {
	            fprintf(stderr,
		            "function_eval error in DiriBC\n");
	            return FAIL;
	          }
	        for (j=0; j<dim;j++)
	          {
	            (*u0).V[node1+j*vx_nr] = feval_result[j];
	            (*P).V[node1+j*vx_nr]=1;
	          }

	        /* NODE 2 Fct-eval and copying to nodem.V */
	        err=mesh_func_eval(&(*msh), fu,
		  	           &(*msh).vertex[node2*(*msh).vx_w+MCT2VXSTRT],
			           dim, feval_result);

	        if ((err)!=SUCCESS)
	          {
	            fprintf(stderr,
		            "function_eval error in DiriBC\n");
	            return FAIL;
	          }

	        for (j=0; j<dim;j++)
	          {
	            (*u0).V[node2+j*vx_nr] = feval_result[j];
	            (*P).V[node2+j*vx_nr]=1;
	          }
 
	        /* NODE 3 Fct-eval and copying to nodem.V */
                err=mesh_func_eval(&(*msh), fu, 
			           &(*msh).vertex[node3*(*msh).vx_w+MCT2VXSTRT],
			           dim, feval_result);

	        if ((err)!=SUCCESS)
	          {
	            fprintf(stderr, 
		            "function_eval error in DiriBC\n");
	            return FAIL;
	      }

	    for (j=0; j<dim;j++)
	      {
	        (*u0).V[node3+j*vx_nr] = feval_result[j];
	        (*P).V[node3+j*vx_nr]=1;
	      }

		free(feval_result);
	      /* at this stage the projector array holds the info if a DOF
	         is Dirichlet
	      */
            }
        } /* end take Dirichlet BC into account */
  
      /* now correct the projector to a list of Dirichlet DOFs */
      j=0;
      for (i=0; i<vx_nr*dim; i++)
        {
          if ((*P).V[i]==1)
	    {
	      /* this is a Dirichlet node, append it to the list */
	      (*P).V[j]=i;
	      j++;
	    }
        }
      (*P).len=j;

      /* Setting Neumann BC */
      err=lame_neum_bc_t2 (msh, rhs );
      FUNCTION_FAILURE_HANDLE( err, lame_neum_bc_t2, assem_lame_tx);
      }
    }

  if ( (*K).type == SP_TYPE_FLEX )
    {
      err = sparse_convert_compressed_row( K );
      FUNCTION_FAILURE_HANDLE( err, sparse_convert_compressed_row, 
			       assem_lame_tx );
    }

  /* free local data */
  free_intdata (&iform);
  free(elK);
  free(Jac);
  free(Jacinv);
  free(Jinvgrad);
  free(ldofs);


  return SUCCESS;
}

/*FUNCTION*/
int lame_neum_bc_t2( struct mesh *m, struct vector *rhs
/* adds the contribution of the Neumann boundary conditions to the rhs
       sigma(u)*n=g on Gamma_N
   ==>
       rhs(v) += int_{Gamma_N} +g*v ds

   Input:  m       - the mesh (2d T2), input

   Output: rhs     - rhighthandside vector, set according to the
                     integral formula above

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  int  err;
  FIDX bd, i, j, k;

  FIDX dim=2, vx_nr, bas_n1d1, vx_w, eg_w, bd_w;

  struct int_data iform1;/* integration formula 1d   */

  FIDX subtypes[1];
  double *phi1d1, *gradp1d1, *rhs_loc;

  vx_nr = (*m).vx_nr; vx_w  = (*m).vx_w;
  eg_w  = (*m).eg_w;  bd_w  = (*m).bd_w;  

  /* get integration formula */
  subtypes[0]=2;
  err=cubature_bases( dim-1, 3, inter, 1, subtypes, &iform1); 
  FUNCTION_FAILURE_HANDLE( err, cubature_bases, lame_neum_bc_t2);

  /* make phi and gradphi better accessible */
  phi1d1   = (iform1.bases[0]->phi);
  gradp1d1 = (iform1.bases[0]->gradphi);
  bas_n1d1 = (iform1.bases[0]->num_basis);

  TRY_MALLOC(rhs_loc, bas_n1d1*dim, double, lame_neum_bc_t2);

  /* loop over boundary, Neumann BC implementation */
  for (bd=0; bd<(*m).bd_nr; bd++)
    {
      /* take Neumann BC into account */
      if (((*m).bound[bd*bd_w + MCT2BDTYPE]==2)&&
	  ((*m).edge[(*m).bound[bd*bd_w+MCT2BDEDGE]*eg_w+MCT2EGCHL1]==-1))
	{
	  FIDX edi, node[3], fu;
	  double g[2], x[2], t[2];
	  double ds;

	  /* check out which nodes */
	  edi   = (*m).bound[bd*bd_w + MCT2BDEDGE];

	  node[0] = (*m).edge[edi*eg_w + MCT2EGNOD1];
	  node[1] = (*m).edge[edi*eg_w + MCT2EGNOD1 +1];
	  node[2] = (*m).edge[edi*eg_w + MCT2EGNODM];
    

	  /* get the set boundary values */
	  fu=(*m).bound[bd*bd_w + MCT2BDFNCT];

	  /* set rhs_loc to zero */
	  for (i=0; i<dim*bas_n1d1; i++)
	    rhs_loc[i]=0.0;


	  /* loop over integration points */
	  for (k=0; k<iform1.num_points; k++)
	    {
	      
	      /* the integration point and tangential */
	      for (i=0; i<dim; i++)
		{
		  x[i]=0;
		  t[i]=0;
		  for (j=0; j<bas_n1d1; j++)
		    {
		      x[i] += (*m).vertex[node[j]*vx_w+MCT2VXSTRT+i]
			*phi1d1[k*bas_n1d1 +j];
		      t[i] += (*m).vertex[node[j]*vx_w+MCT2VXSTRT+i]
			*(gradp1d1[k*bas_n1d1+j]);
		    }
		}

	      err=mesh_func_eval(m, fu, x, 2, g );
	      if (err!=SUCCESS)
		{
		  fprintf(stderr, "lame_neum_bc_t2: "
			  "function_eval error in NeumannBC\n");
		  return FAIL;
		}
	  
	      /* ds= norm(t) */
	      ds=sqrt( t[0]*t[0]+t[1]*t[1] );

	      /* build the integral */
	      for (i=0; i<dim; i++)
		{
		  for (j=0; j<bas_n1d1; j++)
		    {
		      rhs_loc[i*bas_n1d1+j] += 
			g[i]*phi1d1[k*bas_n1d1+j] * ds * iform1.weights[k];
		    }
		}
	    }/* end loop integration points */

	  /* the local rhs is now complete, add it to the global rhs */
	  for (i=0; i<dim; i++)
	    {
	      for (j=0; j<bas_n1d1; j++)
		{
		  (*rhs).V[i*vx_nr+node[j]] =+ rhs_loc[i*bas_n1d1 + j];
		}
	    }
	    
	} /* end this BD is Neumann */
      
    } /* end loop over boundary elements */

  /* free local data */
  free_intdata (&iform1);
  free(rhs_loc);

  return SUCCESS;
}






/*FUNCTION*/
int assem_elem_detA_tx(struct mesh *m, double *elem_detA, int type
/* computes the detA the determinate of the element linear mapping 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: elem_detA-detA for each element

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el,i,j,r;
  int err;

  FIDX dim=2, bas_n1, vx_nr, el_w, vx_w;
  FIDX subtypes[1];

  struct int_data iform; /* integration formula    */
  double *Jac;           /* Jacobian of the element mapping */
  double detJac;         /* determinant of the Jacobian */
  double *phi1, *gradp1;

  FIDX l_mctxELNOD1, l_mctxVXSTRT;

  switch (type)
    {
    case 1:
      l_mctxVXSTRT = MCT1VXSTRT;
      l_mctxELNOD1 = MCT1ELNOD1;
      break;
    case 2:
      l_mctxVXSTRT = MCT2VXSTRT;
      l_mctxELNOD1 = MCT2ELNOD1;
      break;
    default:
      fprintf(stderr,"assem_elem_detA_tx: unknown type=%d\n",type);
      return FAIL;
    }

  /* init */
  /* get integration formula */
  subtypes[0]=1;

  err=cubature_bases( dim, 2, tria, 1, subtypes, &iform); 

  if (err!=SUCCESS)
    {
      fprintf( stderr,
	       "cubature_bases returned error in assem_elem_detA_tx!\n");
      return FAIL;
    }

  /* make phi and gradphi better accessible */
  phi1  = (iform.bases[0]->phi);
  gradp1= (iform.bases[0]->gradphi);
  bas_n1= (iform.bases[0]->num_basis);
  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;

  /* allocate memory for Jac */
  TRY_MALLOC( Jac, dim*dim, double, assem_elem_detA_tx);

  /* loop over all elements */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* compute the Jacobian for this element 
	 (constant on the element)*/
      /* Jac=0 */
      for (i=0; i<dim*dim; i++)
	Jac[i]=0.0;
	  
      /* Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
      for (i=0;i<bas_n1; i++)
	{
	  for (j=0; j<dim; j++)
	    for (r=0; r<dim; r++)
	      {
		Jac[j*dim+r]+= 
		  (*m).vertex[(*m).elem[el*el_w+l_mctxELNOD1+i]*vx_w
			      +l_mctxVXSTRT+j]
		  * gradp1[0*bas_n1*dim +i*dim +r];
	      }
	}

      /* get detJac */
      detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];

      elem_detA[el]=detJac;
    } /* end loop over all elements */


  /* free local data */
  free_intdata (&iform);
  free(Jac);

  return SUCCESS;
}








/*FUNCTION*/
int max_angle_tx(struct mesh *m, double *maxal, int type
/* computes the maximum interior angle amon the triangular elements in
   the mesh 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: maxal   - maximum interior angle for all elements, in
                     radiant (rather than degrees), given by reference

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		    ){
  FIDX el,cn,i;

  FIDX dim=2, vx_nr, el_w, vx_w;

  double *va, *vb;

  FIDX l_mctxELNOD1, l_mctxVXSTRT;

  switch (type)
    {
    case 1:
      l_mctxVXSTRT = MCT1VXSTRT;
      l_mctxELNOD1 = MCT1ELNOD1;
      break;
    case 2:
      l_mctxVXSTRT = MCT2VXSTRT;
      l_mctxELNOD1 = MCT2ELNOD1;
      break;
    default:
      fprintf(stderr,"max_angle_tx: unknown type=%d\n",type);
      return FAIL;
    }

  /* init */
  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;

  *maxal=0.0;

  /* allocate memory for va, vb */
  TRY_MALLOC( va, dim, double, max_angle_tx);
  TRY_MALLOC( vb, dim, double, max_angle_tx);

  /* loop over all elements */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* loop over all corners of the triangle */
      for (cn=0; cn<3; cn++)
	{
	  FIDX ncn, nca, ncb;
	  double nva, nvb, cosal, al;

	  /* define node numbers for this corner */
	  ncn = (*m).elem[el*el_w+l_mctxELNOD1+ cn     ];
	  nca = (*m).elem[el*el_w+l_mctxELNOD1+(cn+1)%3];
	  ncb = (*m).elem[el*el_w+l_mctxELNOD1+(cn+2)%3];

	  /* compute edge vectors va, vb, and their norm */
	  nva = 0.0;
	  nvb = 0.0;
	  for (i=0; i<dim; i++)
	    {
	      va[i]=(*m).vertex[nca*vx_w+l_mctxVXSTRT+i]
		-(*m).vertex[ncn*vx_w+l_mctxVXSTRT+i];
	      vb[i]=(*m).vertex[ncb*vx_w+l_mctxVXSTRT+i]
		-(*m).vertex[ncn*vx_w+l_mctxVXSTRT+i];
	      
	      nva += va[i]*va[i];
	      nvb += vb[i]*vb[i];
	    }

	  if ((nva!=0.0)&&(nvb!=0.0))
	    {
	      /* normalise the vectors, compute the scalar product (va,vb)*/
	      nva = 1.0/sqrt(nva);
	      nvb = 1.0/sqrt(nvb);
	      cosal = 0.0;
	      for (i=0; i<dim; i++)
		{
		  va[i] *= nva;
		  vb[i] *= nvb;

		  cosal += va[i]*vb[i];
		}
	  
	      al=acos(cosal);
	    }
	  else
	    { /* angle between vectors of lenght zero not defined, but
		 we just say it's too large */
	      al= M_PI;
	    }

	  if (al> *maxal) *maxal = al;
	}
    } /* end loop over all elements */


  /* free local data */
  free(va);
  free(vb);

  return SUCCESS;
}
